*-------------------------------------------------------------------------------
/* title : Compute first set of Instrumental Variables (IV1) for intra- and 
		   inter-industry proxies (see equation 7 in paper)		   
   source: World Input-Output Tables, 2016 Release (in Stata format)
		   http://www.wiod.org/database/wiots16	 
   input : $data/wiot_stata_Nov16/WIOT`year'_October16_ROW.dta
   output: $temp/Proxies_IV1.dta */
*-------------------------------------------------------------------------------

*===============================================================================
* Set up
*===============================================================================
*--------------------------SET-UP & READ DIRECTORIES----------------------------
cd												// Current directory of code
do "[0.0] directories.do"

*===============================================================================
* Intra-industry proxies
*===============================================================================

* Create intra-industry importing shares (IM)
qui foreach country in INST {
	forvalues y=2000/2014 {
		use "$data/wiot_stata_Nov16/WIOT`y'_October16_ROW.dta", clear
		drop if Country=="TOT"
		drop TOT *57 *58 *59 *60 *61
		rename RNr id
		drop if id>56
		rename Country country
		rename Year year
		forvalues v=1/56 {
			gen v`country'`v' = vAUS`v' + vCAN`v' + vJPN`v' + vKOR`v' + vTWN`v' + vUSA`v' 
		}
		keep country year IndustryCode IndustryDescription id *`country'*
		gen partner="RoW"
		replace partner="CHN" if country=="CHN"
		replace partner="CEEC" if country=="BGR" | country=="CZE" | country=="EST" | country=="HRV" | country=="HUN" | country=="LTU" | country=="LVA" | country=="POL" | country=="ROU" | country=="SVN" | country=="SVK"		
		replace partner="WEU" if country=="AUT" | country=="BEL" | country=="DEU" | country=="DNK" | country=="ESP" | country=="FIN" | country=="FRA" | country=="GBR" | country=="GRC" | country=="IRL" | country=="ITA" | country=="NLD" | country=="NOR" | country=="PRT" | country=="SWE" 		
		replace partner="`country'" if country=="AUS" | country=="CAN" | country=="JPN" | country=="KOR" | country=="TWN" | country=="USA"

		gcollapse (sum) v`country'1-v`country'56, by(year partner) 
		foreach j of varlist v`country'1-v`country'56 {
			gegen `j'_sum = total(`j'), by(year)
			gen IM_`j' = `j'/(`j'_sum)
			replace IM_`j' = 0 if `j'==0 & `j'_sum==0
			drop `j' `j'_sum
		}
		rename IM_v`country'* IM_iv1_* 
		drop if partner=="`country'"
		greshape long IM_iv1_, i( year partner ) j(industry) string
		greshape wide IM_iv1_, i(year industry) j(partner) string
		gen IM_iv1 = IM_iv1_CEEC+IM_iv1_CHN+IM_iv1_WEU+IM_iv1_RoW
		gen IM_iv1_noCHN = IM_iv1-IM_iv1_CHN
		gen IM_iv1_noCEEC = IM_iv1-IM_iv1_CEEC
		gen IM_iv1_noCHNCEEC = IM_iv1-IM_iv1_CEEC-IM_iv1_CHN
		cap gen country="`country'"
		destring industry, replace force
		keep country year industry IM*
		order country year industry IM_iv1
		sort industry year	
		save "$temp/IM`country'_`y'.dta", replace
	}
	noi dis "`country'"
}

* Create intra-industry exporting shares (EX)
qui foreach country in INST {
	forvalues y=2000/2014 {
		use "$data/wiot_stata_Nov16/WIOT`y'_October16_ROW.dta", clear
		drop if Country=="TOT"
		drop TOT *57 *58 *59 *60 *61
		rename RNr id
		drop if id>56
		rename Country country
		rename Year year
		replace country="`country'" if country=="AUS" | country=="CAN" | country=="JPN" | country=="KOR" | country=="TWN" | country=="USA"
		keep if country=="`country'"
		forvalues v=1/56 {
			gen v`country'`v' = vAUS`v' + vCAN`v' + vJPN`v' + vKOR`v' + vTWN`v' + vUSA`v'
			drop vAUS`v' vCAN`v' vJPN`v' vKOR`v' vTWN`v' vUSA`v'
		}
		gcollapse (sum) v* , by(country id year)

		forvalues j=1/56 {	
			gen v`j'_ttl = 0 
			foreach c in AUT BEL DEU DNK ESP FIN FRA GBR GRC IRL ITA NLD NOR PRT SWE BGR CZE EST HRV HUN LTU LVA POL ROU SVK SVN ///
				CHN BRA CHE CYP IDN IND LUX MEX MLT RUS TUR ROW INST {
				replace v`j'_ttl = v`j'_ttl + v`c'`j' 
			}
			gen v`j'_ttl_RoW = v`j'_ttl - v`country'`j'
			gen v`j'_CEEC = 0
			foreach c in BGR CZE EST HRV HUN LTU LVA POL ROU SVK SVN {
				replace v`j'_CEEC = v`j'_CEEC + v`c'`j' //CEEC-countries
			}
		}
		gen ttl = 0
		gen ttl_RoW = 0
		gen ttl_CEEC = 0
		gen ttl_CHN=0
		forvalues j=1/56 {	
			replace ttl = ttl + v`j'_ttl
			replace ttl_RoW = ttl_RoW + v`j'_ttl_RoW
			replace ttl_CEEC = ttl_CEEC + v`j'_CEEC
			replace ttl_CHN = ttl_CHN + vCHN`j'		
		}
		gen EX_iv1 = ttl_RoW/ttl
		gen EX_iv1_CEEC = ttl_CEEC/ttl
		gen EX_iv1_noCEEC = (ttl_RoW-ttl_CEEC)/ttl
		gen EX_iv1_CHN = ttl_CHN/ttl
		gen EX_iv1_noCHN = (ttl_RoW-ttl_CHN)/ttl
		gen EX_iv1_noCHNCEEC = (ttl_RoW-ttl_CHN-ttl_CEEC)/ttl
		foreach j in "" _CEEC _noCEEC _CHN _noCHN _noCHNCEEC {
			replace EX_iv1`j' = 0 if EX_iv1`j'==.
		}
		keep country year id EX*
		rename id industry
		sort industry year
		save "$temp/EX`country'_`y'.dta", replace
	}
	noi dis "`country'"
}

*===============================================================================
* Inter-industry proxies that include all supply links (broad)
*===============================================================================

* STEP 1: construct "matrices" of weights from domestic IO tables
qui foreach country in AUT BEL DEU DNK ESP FIN FRA GBR GRC IRL ITA NLD NOR PRT SWE BGR CZE EST HRV HUN LTU LVA POL ROU SVK SVN {
	forvalues y=2000/2014 {
		use "$data/wiot_stata_Nov16/WIOT`y'_October16_ROW.dta", clear
		keep if Country=="`country'"
		rename Country country
		rename RNr id
		drop if id>56
		drop *57 *58 *59 *60 *61
		keep country id *`country'*
		rename v`country'* i*

		gegen i_rowsum=rowtotal(i1-i56)
		forvalues v=1/56 {
			gen c_down`v' = i`v'/i_rowsum
			replace c_down`v'=0 if c_down`v'==. 	// need zeros for matrix manipulation later on
		}
		forvalues v=1/56 {
			gegen i`v'_colsum=total(i`v')
			gen c_up`v' = i`v'/i`v'_colsum
			replace c_up`v'=0 if c_up`v'==. 		// need zeros for matrix manipulation later on
			drop i`v'_colsum
		}
		forvalues v=1/56 {
			gen c_down`v'_0D = c_down`v' 
			replace c_down`v'_0D = 0 if id==`v' 
		}
		forvalues v=1/56 {
			gen c_up`v'_0D = c_up`v' 
			replace c_up`v'_0D = 0 if id==`v' 
		}
		rename id bid
		drop i_rowsum i*
		gen year = `y'
		rename bid industry
		order country year industry
		compress
		save "$temp/COEF`country'_`y'.dta", replace
	}
	noi dis "`country'"
}

* STEP 2a: proxies of downIM, upEX, upIM, downEX with time-varying weights
qui foreach country in AUT BEL DEU DNK ESP FIN FRA GBR GRC IRL ITA NLD NOR PRT SWE BGR CZE EST HRV HUN LTU LVA POL ROU SVK SVN {
	forvalues y=2000/2014 {
		use "$temp/IMINST_`y'.dta", clear
		merge 1:1 country industry year using "$temp/EXINST_`y'.dta"
		drop _merge
		drop country
		merge 1:1 industry year using "$temp/COEF`country'_`y'.dta"
		drop _merge
		order country industry year 
		
		mkmat IM_iv1, matrix(IM)
		mkmat IM_iv1_CEEC, matrix(IMCEEC) 
		mkmat IM_iv1_CHN, matrix(IMCHN)
		mkmat EX_iv1, matrix(EX)
		mkmat EX_iv1_CEEC, matrix(EXCEEC)
		mkmat EX_iv1_CHN, matrix(EXCHN)
		mkmat c_down1-c_down56, matrix(DOWN)
		mkmat c_up1-c_up56, matrix(UP)
		mkmat c_down1_0D-c_down56_0D, matrix(DOWNZD)
		mkmat c_up1_0D-c_up56_0D, matrix(UPZD)

		foreach type in IM IMCEEC IMCHN EX EXCEEC EXCHN {
			mat dwn0d`type' = DOWNZD*`type'
			svmat dwn0d`type', names(down0d`type')
			rename down0d`type'* down0d`type'_iv1
			mat upp0d`type' = UPZD'*`type'
			svmat upp0d`type', names(up0d`type')
			rename up0d`type'* up0d`type'_iv1
			mat dwn`type' = DOWN*`type'
			svmat dwn`type', names(down`type')
			rename down`type'* down`type'_iv1
			mat upp`type' = UP'*`type'
			svmat upp`type', names(up`type')
			rename up`type'* up`type'_iv1
		}
		keep country industry year *_iv1 IM_iv1_CHN EX_iv1_CHN
		foreach var in down0dIM up0dEX down0dIMCEEC up0dEXCEEC down0dIMCHN up0dEXCHN ///
			downIM upEX downIMCEEC upEXCEEC downIMCHN upEXCHN {
			rename `var'_iv1 `var'broad_iv1
		}
		save "$temp/IM`country'_`y'_TV.dta", replace
	}
	noi dis "`country'"
}

* Append data
qui foreach country in AUT BEL DEU DNK ESP FIN FRA GBR GRC IRL ITA NLD NOR PRT SWE BGR CZE EST HRV HUN LTU LVA POL ROU SVK SVN {
	use "$temp/IM`country'_2000_TV.dta", replace
	forvalues y=2001/2014 {
		append using "$temp/IM`country'_`y'_TV.dta"
	}
	sort country industry year
	save "$temp/Proxies_`country'.dta", replace
}

use "$temp/Proxies_AUT.dta", clear
foreach country in BEL DEU DNK ESP FIN FRA GBR GRC IRL ITA NLD NOR PRT SWE BGR CZE EST HRV HUN LTU LVA POL ROU SVK SVN {
	append using "$temp/Proxies_`country'.dta"
}
sort country industry year
save "$temp/Proxies_IV1.dta", replace

* STEP 2b: proxies of downIM, upEX, upIM, downEX with base-year (2000) weights
qui foreach country in AUT BEL DEU DNK ESP FIN FRA GBR GRC IRL ITA NLD NOR PRT SWE BGR CZE EST HRV HUN LTU LVA POL ROU SVK SVN {
	forvalues y=2000/2014 {
		use "$temp/IMINST_`y'.dta", clear
		merge 1:1 country industry year using "$temp/EXINST_`y'.dta"
		drop _merge
		drop country
		merge 1:1 industry year using "$temp/COEF`country'_`y'.dta"
		drop _merge
		order country industry year 
		
		mkmat IM_iv1, matrix(IM)
		mkmat IM_iv1_CEEC, matrix(IMCEEC) 
		mkmat IM_iv1_CHN, matrix(IMCHN)
		mkmat EX_iv1, matrix(EX)
		mkmat EX_iv1_CEEC, matrix(EXCEEC)
		mkmat EX_iv1_CHN, matrix(EXCHN)
		mkmat c_down1-c_down56, matrix(DOWN)
		mkmat c_up1-c_up56, matrix(UP)
		mkmat c_down1_0D-c_down56_0D, matrix(DOWNZD)
		mkmat c_up1_0D-c_up56_0D, matrix(UPZD)

		foreach type in IM IMCEEC IMCHN EX EXCEEC EXCHN {
			mat dwn0d`type' = DOWNZD*`type'
			svmat dwn0d`type', names(down0d`type')
			rename down0d`type'* down0d`type'_00_iv1
			mat upp0d`type' = UPZD'*`type'
			svmat upp0d`type', names(up0d`type')
			rename up0d`type'* up0d`type'_00_iv1
			mat dwn`type' = DOWN*`type'
			svmat dwn`type', names(down`type')
			rename down`type'* down`type'_00_iv1
			mat upp`type' = UP'*`type'
			svmat upp`type', names(up`type')
			rename up`type'* up`type'_00_iv1
		}
		keep country industry year down* up* IM_iv1_CHN EX_iv1_CHN
		foreach var in down0dIM up0dEX down0dIMCEEC up0dEXCEEC down0dIMCHN up0dEXCHN ///
			downIM upEX downIMCEEC upEXCEEC downIMCHN upEXCHN {
			rename `var'_00_iv1 `var'_00broad_iv1
		}
		keep country industry year *_iv1
		save "$temp/IM`country'_`y'_00.dta", replace
	}
	noi dis "`country'"
}

* Append data
qui foreach country in AUT BEL DEU DNK ESP FIN FRA GBR GRC IRL ITA NLD NOR PRT SWE BGR CZE EST HRV HUN LTU LVA POL ROU SVK SVN {
	use "$temp/IM`country'_2000_00.dta", replace
	forvalues y=2001/2014 {
		append using "$temp/IM`country'_`y'_00.dta"
	}
	sort country industry year
	save "$temp/Proxies_`country'_00.dta", replace
	
	use "$temp/Proxies_`country'.dta", clear
	merge 1:1 country industry year using "$temp/Proxies_`country'_00.dta"
	drop _merge
	save "$temp/Proxies_`country'.dta", replace
}

use "$temp/Proxies_AUT_00.dta", clear
qui foreach country in BEL DEU DNK ESP FIN FRA GBR GRC IRL ITA NLD NOR PRT SWE BGR CZE EST HRV HUN LTU LVA POL ROU SVK SVN {
	append using "$temp/Proxies_`country'_00.dta"
}
sort country industry year
save "$temp/Proxies_00.dta", replace

use "$temp/Proxies_IV1.dta", clear
merge 1:1 country industry year using "$temp/Proxies_00.dta"
drop _merge
save "$temp/Proxies_IV1.dta", replace

* Housekeeping
qui foreach country in AUT BEL DEU DNK ESP FIN FRA GBR GRC IRL ITA NLD NOR PRT SWE BGR CZE EST HRV HUN LTU LVA POL ROU SVK SVN {
	forvalues y=2000/2014 {
		cap erase "$temp/IM`country'_`y'.dta"
		cap erase "$temp/IM`country'_`y'_TV.dta"
		cap erase "$temp/IM`country'_`y'_00.dta"
		cap erase "$temp/COEF`country'_`y'.dta"
		cap erase "$temp/EX`country'_`y'.dta"
		cap erase "$temp/Proxies_`country'_00.dta"
		cap erase "$temp/Proxies_00.dta"
	}
}

*===============================================================================
* Additional inter-industry proxies based on the observed & not all supply links
*===============================================================================

* Create intra-industry importing shares (IM)
qui foreach country in INST {
	forvalues y=2000/2014 {
		use "$data/wiot_stata_Nov16/WIOT`y'_October16_ROW.dta", clear
		drop if Country=="TOT"
		drop TOT *57 *58 *59 *60 *61
		rename RNr id
		drop if id>56
		rename Country country
		rename Year year
		forvalues v=1/56 {
			gen v`country'`v' = vAUS`v' + vCAN`v' + vJPN`v' + vKOR`v' + vTWN`v' + vUSA`v' 
		}
		keep country year IndustryCode IndustryDescription id *`country'*
		gen partner="RoW"
		replace partner="CHN" if country=="CHN" 
		replace partner="CEEC" if country=="BGR" | country=="CZE" | country=="EST" | country=="HRV" | country=="HUN" | country=="LTU" | country=="LVA" | country=="POL" | country=="ROU" | country=="SVN" | country=="SVK" 		
		replace partner="WEU" if country=="AUT" | country=="BEL" | country=="DEU" | country=="DNK" | country=="ESP" | country=="FIN" | country=="FRA" | country=="GBR" | country=="GRC" | country=="IRL" | country=="ITA" | country=="NLD" | country=="NOR" | country=="PRT" | country=="SWE" 		
		replace partner="`country'" if country=="AUS" | country=="CAN" | country=="JPN" | country=="KOR" | country=="TWN" | country=="USA" 

		rename id prod
		gcollapse (sum) v`country'1-v`country'56, by(year prod partner) 
		foreach v of varlist v`country'1-v`country'56 { // v is industry that imports "product" id (supplier-industry)
			gegen `v'_sum = total(`v'), by(year prod)
			gen IM_`v' = `v'/(`v'_sum)
			replace IM_`v' = 0 if `v'==0 & `v'_sum==0
			drop `v' `v'_sum
		}
		rename IM_v`country'* IM_* 
		drop if partner=="`country'"
		greshape long IM_, i( year prod partner) j(industry) string
		tostring prod, gen(partnerprod)
		forvalues v=1/9 {
			replace partnerprod ="0`v'" if partnerprod=="`v'"
		}
		replace partnerprod = partner + partnerprod
		drop partner prod
		greshape wide IM_, i(year industry) j(partnerprod) string

		foreach v in 01 02 03 04 05 06 07 08 09 10 11 12 13 14 15 16 17 18 19 20 21 22 23 24 25 26 27 28 29 30 31 32 33 34 35 36 37 38 39 40 41 42 43 44 45 46 47 48 49 50 51 52 53 54 55 56 {
			gen IM`v' = IM_CEEC`v'+IM_CHN`v'+IM_WEU`v'+IM_RoW`v'
		}
		foreach v in 01 02 03 04 05 06 07 08 09 10 11 12 13 14 15 16 17 18 19 20 21 22 23 24 25 26 27 28 29 30 31 32 33 34 35 36 37 38 39 40 41 42 43 44 45 46 47 48 49 50 51 52 53 54 55 56 {
			gen IM_noCHN`v' = IM`v'-IM_CHN`v'
		}
		foreach v in 01 02 03 04 05 06 07 08 09 10 11 12 13 14 15 16 17 18 19 20 21 22 23 24 25 26 27 28 29 30 31 32 33 34 35 36 37 38 39 40 41 42 43 44 45 46 47 48 49 50 51 52 53 54 55 56 {
			gen IM_noCEEC`v' = IM`v'-IM_CEEC`v'
		}
		foreach v in 01 02 03 04 05 06 07 08 09 10 11 12 13 14 15 16 17 18 19 20 21 22 23 24 25 26 27 28 29 30 31 32 33 34 35 36 37 38 39 40 41 42 43 44 45 46 47 48 49 50 51 52 53 54 55 56 {
			gen IM_noCHNCEEC`v' = IM`v'-IM_CEEC`v'-IM_CHN`v'
		}
		gen country="`country'"
		destring industry, replace force
		keep country year industry IM*
		order country
		sort industry year	
		save "$temp/IM`country'_`y'.dta", replace
	}
	noi dis "`country'"
}	

* Create intra-industry exporting shares (EX)
qui foreach country in INST {
	forvalues y=2000/2014 {
		use "$data/wiot_stata_Nov16/WIOT`y'_October16_ROW.dta", clear
		drop if Country=="TOT"
		drop TOT *57 *58 *59 *60 *61
		rename RNr id
		drop if id>56
		rename Country country
		rename Year year
		replace country="`country'" if country=="AUS" | country=="CAN" | country=="JPN" | country=="KOR" | country=="TWN" | country=="USA"
		keep if country=="`country'"
		gcollapse (sum) v* , by(country id year)
		forvalues v=1/56 {
			gen v`country'`v' = vAUS`v' + vCAN`v' + vJPN`v' + vKOR`v' + vTWN`v' + vUSA`v' 
		}
		forvalues v=1/56 {	
			gen v`v'_ttl = 0 
			foreach c in AUT BEL DEU DNK ESP FIN FRA GBR GRC IRL ITA NLD NOR PRT SWE BGR CZE EST HRV HUN LTU LVA POL ROU SVK SVN ///
				CHN BRA CHE CYP IDN IND LUX MEX MLT RUS TUR ROW INST { 
				replace v`v'_ttl = v`v'_ttl + v`c'`v' 
			}
			gen v`v'_ttl_RoW = v`v'_ttl - v`country'`v'
			gen v`v'_CEEC = 0
			foreach c in BGR CZE EST HRV HUN LTU LVA POL ROU SVK SVN {
				replace v`v'_CEEC = v`v'_CEEC + v`c'`v' //CEEC-countries
				replace v`v'_CEEC = v`v'_CEEC- v`c'`v' if country=="`c'"
			}
		}
		forvalues v=1/56 {				
			gen EX`v' = v`v'_ttl_RoW/v`v'_ttl
			replace EX`v'=0 if EX`v'==.
		}
		forvalues v=1/56 {	
			gen EX`v'_CEEC = v`v'_CEEC/v`v'_ttl
			replace EX`v'_CEEC=0 if EX`v'_CEEC==.
		}
		forvalues v=1/56 {	
			gen EX`v'_CHN = vCHN`v'/v`v'_ttl
			replace EX`v'_CHN=0 if EX`v'_CHN==.
		}
		forvalues v=1/56 {	
			gen EX`v'_noCEEC = (v`v'_ttl_RoW-v`v'_CEEC)/v`v'_ttl
			replace EX`v'_noCEEC=0 if EX`v'_noCEEC==.
		}
		forvalues v=1/56 {	
			gen EX`v'_noCHN = (v`v'_ttl_RoW-vCHN`v')/v`v'_ttl
			replace EX`v'_noCHN=0 if EX`v'_noCHN==.
		}
		forvalues v=1/56 {	
			gen EX`v'_noCHNCEEC = (v`v'_ttl_RoW-vCHN`v'-v`v'_CEEC)/v`v'_ttl
			replace EX`v'_noCHNCEEC=0 if EX`v'_noCHNCEEC==.
		}
		keep country year id EX*
		rename id industry
		sort industry year
		save "$temp/EX`country'_`y'.dta", replace
	}
	noi dis "`country'"
}

* STEP 1: construct "matrices" of weights from domestic IO tables
qui foreach country in AUT BEL DEU DNK ESP FIN FRA GBR GRC IRL ITA NLD NOR PRT SWE BGR CZE EST HRV HUN LTU LVA POL ROU SVK SVN {
	forvalues y=2000/2014 {
		use "$data/wiot_stata_Nov16/WIOT`y'_October16_ROW.dta", clear
		keep if Country=="`country'"
		rename Country country
		rename RNr id
		drop if id>56
		drop *57 *58 *59 *60 *61
		keep country id *`country'*
		rename v`country'* i*

		gegen i_rowsum=rowtotal(i1-i56)
		forvalues v=1/56 {
			gen c_down`v' = i`v'/i_rowsum
			replace c_down`v'=0 if c_down`v'==. 	// need zeros for matrix manipulation later on
		}
		forvalues v=1/56 {
			gegen i`v'_colsum=total(i`v')
			gen c_up`v' = i`v'/i`v'_colsum
			replace c_up`v'=0 if c_up`v'==. 		// need zeros for matrix manipulation later on
			drop i`v'_colsum
		}
		forvalues v=1/56 {
			gen c_down`v'_0D = c_down`v' 
			replace c_down`v'_0D = 0 if id==`v' 
		}
		forvalues v=1/56 {
			gen c_up`v'_0D = c_up`v' 
			replace c_up`v'_0D = 0 if id==`v' 
		}
		rename id bid
		drop i_rowsum i*
		gen year = `y'
		rename bid industry
		order country year industry
		compress
		save "$temp/COEF`country'_`y'.dta", replace
	}
	noi dis "`country'"
}

* STEP 2a: proxies of downIM, upEX, upIM, downEX with time-varying weights
qui foreach country in AUT BEL DEU DNK ESP FIN FRA GBR GRC IRL ITA NLD NOR PRT SWE BGR CZE EST HRV HUN LTU LVA POL ROU SVK SVN {
	forvalues y=2000/2014 {
		use "$temp/IMINST_`y'.dta", clear
		merge 1:1 country industry year using "$temp/EXINST_`y'.dta"
		drop _merge
		cap drop country
		merge 1:1 industry year using "$temp/COEF`country'_`y'.dta"
		drop _merge
		order country industry year 
		
		mkmat IM01-IM56, matrix(IM)
		mkmat IM_CEEC01-IM_CEEC56, matrix(IMCEEC) 
		mkmat IM_CHN01-IM_CHN56, matrix(IMCHN)
		mkmat EX1-EX56, matrix(EX)
		mkmat EX1_CEEC-EX56_CEEC, matrix(EXCEEC) 
		mkmat EX1_CHN-EX56_CHN, matrix(EXCHN)
		mkmat c_down1-c_down56, matrix(DOWN)
		mkmat c_up1-c_up56, matrix(UP)
		mkmat c_down1_0D-c_down56_0D, matrix(DOWNZD)
		mkmat c_up1_0D-c_up56_0D, matrix(UPZD)

		foreach type in IM IMCEEC IMCHN {
			mat dwn0d`type' = DOWNZD*`type'
			svmat dwn0d`type', names(down0d`type')
			mat dwn`type' = DOWN*`type'
			svmat dwn`type', names(down`type')
		}
		foreach type in EX EXCEEC EXCHN {
			mat upp0d`type' = UPZD'*`type'
			svmat upp0d`type', names(up0d`type')
			mat upp`type' = UP'*`type'
			svmat upp`type', names(up`type')
		}
		foreach var in down0dIM down0dIMCEEC down0dIMCHN downIM downIMCEEC downIMCHN ///
						up0dEX up0dEXCEEC up0dEXCHN upEX upEXCEEC upEXCHN {
			gen `var'_iv1 = .
		}
		
		forvalues v=1/56 {
			foreach var in down0dIM down0dIMCEEC down0dIMCHN downIM downIMCEEC downIMCHN ///
							up0dEX up0dEXCEEC up0dEXCHN upEX upEXCEEC upEXCHN {
				replace `var'_iv1 = `var'`v' if industry==`v' // need diagonal elements
				drop `var'`v' 
			}
		}
		keep country industry year *_iv1
		save "$temp/IM`country'_`y'_TV.dta", replace
	}
	noi dis "`country'"
}

* Append data
qui foreach country in AUT BEL DEU DNK ESP FIN FRA GBR GRC IRL ITA NLD NOR PRT SWE BGR CZE EST HRV HUN LTU LVA POL ROU SVK SVN {
	use "$temp/IM`country'_2000_TV.dta", replace
	forvalues y=2001/2014 {
		append using "$temp/IM`country'_`y'_TV.dta"
	}
	sort country industry year
	save "$temp/DOWNIMEXUP_`country'.dta", replace
}

use "$temp/DOWNIMEXUP_AUT.dta", clear
qui foreach country in BEL DEU DNK ESP FIN FRA GBR GRC IRL ITA NLD NOR PRT SWE BGR CZE EST HRV HUN LTU LVA POL ROU SVK SVN {
	append using "$temp/DOWNIMEXUP_`country'.dta"
}
sort country industry year
save "$temp/DOWNIMEXUP_IV1.dta", replace

* STEP 2b: proxies of downIM, upEX, upIM, downEX with base-year (2000) weights
qui foreach country in AUT BEL DEU DNK ESP FIN FRA GBR GRC IRL ITA NLD NOR PRT SWE BGR CZE EST HRV HUN LTU LVA POL ROU SVK SVN {
	forvalues y=2000/2014 {
		use "$temp/IMINST_`y'.dta", clear
		merge 1:1 country industry year using "$temp/EXINST_`y'.dta"
		drop _merge
		cap drop country
		merge 1:1 industry using "$temp/COEF`country'_2000.dta"
		drop _merge
		order country industry year 
		
		mkmat IM01-IM56, matrix(IM)
		mkmat IM_CEEC01-IM_CEEC56, matrix(IMCEEC) 
		mkmat IM_CHN01-IM_CHN56, matrix(IMCHN)
		mkmat EX1-EX56, matrix(EX)
		mkmat EX1_CEEC-EX56_CEEC, matrix(EXCEEC) 
		mkmat EX1_CHN-EX56_CHN, matrix(EXCHN)
		mkmat c_down1-c_down56, matrix(DOWN)
		mkmat c_up1-c_up56, matrix(UP)
		mkmat c_down1_0D-c_down56_0D, matrix(DOWNZD)
		mkmat c_up1_0D-c_up56_0D, matrix(UPZD)

		foreach type in IM IMCEEC IMCHN {
			mat dwn0d`type' = DOWNZD*`type'
			svmat dwn0d`type', names(down0d`type')
			mat dwn`type' = DOWN*`type'
			svmat dwn`type', names(down`type')
		}
		foreach type in EX EXCEEC EXCHN {
			mat upp0d`type' = UPZD'*`type'
			svmat upp0d`type', names(up0d`type')
			mat upp`type' = UP'*`type'
			svmat upp`type', names(up`type')
		}
		foreach var in down0dIM down0dIMCEEC down0dIMCHN downIM downIMCEEC downIMCHN ///
						up0dEX up0dEXCEEC up0dEXCHN upEX upEXCEEC upEXCHN {
			gen `var'_00_iv1 = .
		}
		
		forvalues v=1/56 {
			foreach var in down0dIM down0dIMCEEC down0dIMCHN downIM downIMCEEC downIMCHN ///
							up0dEX up0dEXCEEC up0dEXCHN upEX upEXCEEC upEXCHN {
				replace `var'_00_iv1 = `var'`v' if industry==`v'
				drop `var'`v' 
			}
		}
		keep country industry year *_iv1
		save "$temp/IM`country'_`y'_00.dta", replace
	}
	noi dis "`country'"
}

* Append data
qui foreach country in AUT BEL DEU DNK ESP FIN FRA GBR GRC IRL ITA NLD NOR PRT SWE BGR CZE EST HRV HUN LTU LVA POL ROU SVK SVN {
	use "$temp/IM`country'_2000_00.dta", replace
	forvalues y=2001/2014 {
		append using "$temp/IM`country'_`y'_00.dta"
	}
	sort country industry year
	save "$temp/DOWNIMEXUP_`country'_00.dta", replace
	
	use "$temp/DOWNIMEXUP_`country'.dta", clear
	merge 1:1 country industry year using "$temp/DOWNIMEXUP_`country'_00.dta"
	drop _merge
	save "$temp/DOWNIMEXUP_`country'.dta", replace
}

use "$temp/DOWNIMEXUP_AUT_00.dta", clear
qui foreach country in BEL DEU DNK ESP FIN FRA GBR GRC IRL ITA NLD NOR PRT SWE BGR CZE EST HRV HUN LTU LVA POL ROU SVK SVN {
	append using "$temp/DOWNIMEXUP_`country'_00.dta"
}
sort country industry year
save "$temp/DOWNIMEXUP_00_IV1.dta", replace

use "$temp/DOWNIMEXUP_IV1.dta", clear
merge 1:1 country industry year using "$temp/DOWNIMEXUP_00_IV1.dta"
drop _merge
save "$temp/DOWNIMEXUP_IV1.dta", replace

* Merge proxies
use "$temp/Proxies_IV1.dta", clear
merge 1:1 country industry year using "$temp/DOWNIMEXUP_IV1.dta"
drop _merge
order country industry year IM_iv1 EX_iv1 down0dIM_iv1 up0dIM_iv1 down0dEX_iv1 up0dEX_iv1 down0dIMbroad_iv1 ///
	up0dEXbroad_iv1 downIM_iv1 upIM_iv1 downEX_iv1 upEX_iv1 downIMbroad_iv1 upEXbroad_iv1 down0dIM_00_iv1 up0dIM_00_iv1 ///
	down0dEX_00_iv1 up0dEX_00_iv1 down0dIM_00broad_iv1 up0dEX_00broad_iv1 downIM_00_iv1 upIM_00_iv1 downEX_00_iv1 upEX_00_iv1 ///
	downIM_00broad_iv1 upEX_00broad_iv1
compress
save "$temp/Proxies_IV1.dta", replace

* Housekeeping
qui foreach country in AUT BEL DEU DNK ESP FIN FRA GBR GRC IRL ITA NLD NOR PRT SWE BGR CZE EST HRV HUN LTU LVA POL ROU SVK SVN INST {
	forvalues y=2000/2014 {
		cap erase "$temp/IM`country'_`y'.dta"
		cap erase "$temp/IM`country'_`y'_TV.dta"
		cap erase "$temp/IM`country'_`y'_00.dta"
		cap erase "$temp/COEF`country'_`y'.dta"
		cap erase "$temp/EX`country'_`y'.dta"
		cap erase "$temp/DOWNIMEXUP_`country'_00.dta"
		cap erase "$temp/DOWNIMEXUP_00_IV1.dta"
		cap erase "$temp/DOWNIMEXUP_`country'.dta"
		cap erase "$temp/DOWNIMEXUP_IV1.dta"
		cap erase "$temp/Proxies_`country'.dta"
	}
}
